#include "validation.h"

typedef long long ll;

// Check the grammar of the input files.
// You should also check properties of the input.
// E.g., check that a graph is connected.

const ll maxXY = 100'000'000;

int main(int argc, char *argv[]) {
    InputValidator v(argc, argv);
    int n = v.read_integer("n", 2, 200'000);
    v.newline();

   // Map from (x//400, y//400) to list of (i, x, y, diam)
    std::map<std::pair<ll, ll>, std::vector<std::tuple<ll, ll, ll, ll>>> models;

    std::set<int> valid_diameters = {25, 28, 32, 40, 50, 65, 80, 90, 100, 130, 160};

    for (int i = 0; i < n; i++) {
        ll x = v.read_integer("x", 13, maxXY - 13);
        v.space();
        ll y = v.read_integer("y", 13, maxXY - 13);
        v.space();
        int diam = v.read_integer("diam", 25, 160);
        v.newline();

        // Check if diameter is in valid list
        v.check(valid_diameters.contains(diam), "Unexpected base diameter: ", diam);

        // Check bounds: diam/2 <= x and x + diam/2 <= 10^8
        v.check(diam / 2.0 <= x, "x-d out of bounds: was ", x - diam);
        v.check(x + diam / 2.0 <= maxXY, "x+d out of bounds: was ", x + diam);

        // Check bounds: diam/2 <= y and y + diam/2 <= 10^8
        v.check(diam / 2.0 <= y, "y-d out of bounds: was ", y - diam);
        v.check(y + diam / 2.0 <= maxXY, "y+d out of bounds: was ", y + diam);

        // Add to spatial map
        models[{x / 400, y / 400}].push_back({i, x, y, diam});
    }

    // Check for overlapping models
    for (const auto& [key, _] : models) {
        auto [x, y] = key;

        // Collect all models in current bucket and neighboring buckets
        std::set<std::tuple<ll, ll, ll, ll>> bucket;

        for (int dx = -1; dx <= 1; dx++) {
            for (int dy = -1; dy <= 1; dy++) {
                auto neighbor = std::make_pair(x + dx, y + dy);
                if (models.contains(neighbor)) {
                    for (const auto& model : models[neighbor]) {
                        bucket.insert(model);
                    }
                }
            }
        }

        // Check all pairs in the bucket for overlaps
        for (const auto& a : bucket) {
            for (const auto& b : bucket) {
                auto [ai, ax, ay, adiam] = a;
                auto [bi, bx, by, bdiam] = b;

                if (ai == bi) continue;

                // Check if models overlap, which happens when:
                // ((ax - bx) ** 2 + (ay - by) ** 2) * 4 < (adiam + bdiam) ** 2
                ll dx_sq = (ax - bx) * (ax - bx);
                ll dy_sq = (ay - by) * (ay - by);
                ll sum_diam = adiam + bdiam;

                v.check((dx_sq + dy_sq) * 4 >= sum_diam * sum_diam, "Overlapping models ", ai, " and ", bi);
            }
        }
    }
}
